/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */
"use strict";

const {
  createClass,
  createFactory,
  DOM: dom,
  PropTypes
} = require("devtools/client/shared/vendor/react");
const ReactDOM = require("devtools/client/shared/vendor/react-dom");
const { connect } = require("devtools/client/shared/vendor/react-redux");

const { getAllMessages, getAllMessagesUiById } = require("devtools/client/webconsole/new-console-output/selectors/messages");
const MessageContainer = createFactory(require("devtools/client/webconsole/new-console-output/components/message-container").MessageContainer);

const ConsoleOutput = createClass({

  propTypes: {
    jsterm: PropTypes.object.isRequired,
    messages: PropTypes.object.isRequired,
    sourceMapService: PropTypes.object,
    onViewSourceInDebugger: PropTypes.func.isRequired,
  },

  displayName: "ConsoleOutput",

  componentWillUpdate() {
    let node = ReactDOM.findDOMNode(this);
    if (node.lastChild) {
      this.shouldScrollBottom = isScrolledToBottom(node.lastChild, node);
    }
  },

  componentDidUpdate() {
    if (this.shouldScrollBottom) {
      let node = ReactDOM.findDOMNode(this);
      node.scrollTop = node.scrollHeight;
    }
  },

  render() {
    let {
      dispatch,
      messages,
      messagesUi,
      sourceMapService,
      onViewSourceInDebugger
    } = this.props;

    let messageNodes = messages.map(function (message) {
      return (
        MessageContainer({
          dispatch,
          message,
          key: message.id,
          sourceMapService,
          onViewSourceInDebugger,
          open: messagesUi.includes(message.id)
        })
      );
    });
    return (
      dom.div({className: "webconsole-output"}, messageNodes)
    );
  }
});

function isScrolledToBottom(outputNode, scrollNode) {
  let lastNodeHeight = outputNode.lastChild ?
                       outputNode.lastChild.clientHeight : 0;
  return scrollNode.scrollTop + scrollNode.clientHeight >=
         scrollNode.scrollHeight - lastNodeHeight / 2;
}

function mapStateToProps(state) {
  return {
    messages: getAllMessages(state),
    messagesUi: getAllMessagesUiById(state)
  };
}

module.exports = connect(mapStateToProps)(ConsoleOutput);
