/*
*/

const Cc = Components.classes;
const Ci = Components.interfaces;

const PREF_EM_UPDATE_ENABLED = "extensions.update.enabled";

Components.utils.import("resource://gre/modules/Services.jsm");

var EXPORTED_SYMBOLS = [ "AddonManager", "AddonManagerInternal" ];

// A list of providers to load by default
const PROVIDERS = [
  "resource://gre/modules/XPIProvider.jsm",
  "resource://gre/modules/PluginProvider.jsm",
  "resource://gre/modules/LightweightThemeManager.jsm"
];

/**
 * Logs a debugging message.
 * @param   str
 *          The string to log
 */
function LOG(str) {
  dump("*** addons.manager: " + str + "\n");
}

/**
 * Logs a warning message.
 * @param   str
 *          The string to log
 */
function WARN(str) {
  LOG(str);
}

/**
 * Logs an error message.
 * @param   str
 *          The string to log
 */
function ERROR(str) {
  LOG(str);
}

/**
 * Calls a callback method consuming any thrown exception. Any parameters after
 * the callback parameter will be passed to the callback.
 * @param   callback
 *          The callback method to call
 * @returns The return from the callback
 */
function safeCall(callback) {
  var args = Array.slice(arguments, 1);
  try {
    return callback.apply(null, args);
  }
  catch (e) {
    WARN("Exception calling callback: " + e);
  }
}

/**
 * Calls a method on a provider if it exists and consumes any thrown exception.
 * Any parameters after the dflt parameter are passed to the provider's method.
 * @param   provider
 *          The provider to call
 * @param   method
 *          The method name to call
 * @param   dflt
 *          A default return value of the provider does not implement the named
 *          method or throws an error.
 * @returns The nsIRDFContainer, initialized at the root.
 */
function callProvider(provider, method, dflt) {
  if (!(method in provider))
    return dflt;
  var args = Array.slice(arguments, 3);
  try {
    return provider[method].apply(provider, args);
  }
  catch (e) {
    ERROR("Exception calling provider." + method + ": " + e);
    return dflt;
  }
}

/**
 * This is the real manager, kept here rather than in AddonManager to keep its
 * contents hidden from API users.
 */
var AddonManagerPrivate = {
  installListeners: null,
  addonListeners: null,
  providers: [],
  started: false,

  /**
   * Initialises the AddonManager, loading any known providers and initialising
   * them. 
   */
  startup: function AM_startup() {
    if (this.started)
      return;

    this.installListeners = [];
    this.addonListeners = [];

    let appChanged = true;
    try {
      appChanged = Services.appinfo.version != Services.prefs.getCharPref("extensions.lastAppVersion");
    }
    catch (e) { }
    if (appChanged) {
      LOG("Application has been upgraded");
      Services.prefs.setCharPref("extensions.lastAppVersion", Services.appinfo.version);
    }

    // Ensure all known providers have had a chance to register themselves
    PROVIDERS.forEach(function(url) {
      try {
        Components.utils.import(url, {});
      }
      catch (e) {
        ERROR("Exception loading provider \"" + url + "\": " + e);
      }
    });

    let needsRestart = false;
    this.providers.forEach(function(provider) {
      callProvider(provider, "startup");
      if (callProvider(provider, "checkForChanges", false, appChanged))
        needsRestart = true;
    });
    this.started = true;

    // Flag to the platform that a restart is necessary
    if (needsRestart) {
      let appStartup = Cc["@mozilla.org/toolkit/app-startup;1"].
                       getService(Ci.nsIAppStartup2);
      appStartup.needsRestart = needsRestart;
    }
  },

  /**
   * Registers a new AddonProvider.
   * @param   provider
   *          The provider to register
   */
  registerProvider: function AM_registerProvider(provider) {
    this.providers.push(provider);

    // If we're registering after startup call this provider's startup immediately.
    if (this.started)
      callProvider(provider, "startup");
  },

  /**
   * Shuts down the addon manager and all registered providers, this must clean
   * up everything in order for automated tests to fake restarts.
   */
  shutdown: function AM_shutdown() {
    this.providers.forEach(function(provider) {
      callProvider(provider, "shutdown");
    });

    this.installListeners = null;
    this.addonListeners = null;
    this.started = false;
  },

  /**
   * Performs a background update check by starting an update for all add-ons
   * that are available to update.
   */
  backgroundUpdateCheck: function AM_backgroundUpdateCheck() {
    if (!Services.prefs.getBoolPref(PREF_EM_UPDATE_ENABLED))
      return;
    this.getAddonsByType(null, function(addons) {
      addons.forEach(function(addon) {
        if (addon.permissions & AddonManager.PERM_CANUPGRADE) {
          addon.findUpdates({
            onUpdateAvailable: function(addon, install) {
              install.install();
            }
          }, AddonManager.UPDATE_WHEN_PERIODIC_UPDATE);
        }
      });
    });
  },

  /**
   * Calls all registered InstallListeners with an event. Any parameters after
   * the additional parameter are passed to the listener.
   * @param   method
   *          The method on the listeners to call
   * @param   additional
   *          An array of InstallListeners to also call
   * @returns false if any of the listeners returned false, true otherwise
   */
  callInstallListeners: function AM_callInstallListeners(method, additional) {
    let result = true;
    let listeners = this.installListeners;
    if (additional)
      listeners = additional.concat(listeners);
    let args = Array.slice(arguments, 2);
    listeners.forEach(function(listener) {
      try {
        if (method in listener) {
          if (listener[method].apply(listener, args) === false)
            result = false;
        }
      }
      catch (e) { }
    });
    return result;
  },

  /**
   * Calls all registered AddonListeners with an event. Any parameters after
   * the method parameter are passed to the listener.
   * @param   method
   *          The method on the listeners to call
   */
  callAddonListeners: function AM_callInstallListeners(method) {
    var args = Array.slice(arguments, 1);
    this.addonListeners.forEach(function(listener) {
      try {
        if (method in listener)
          listener[method].apply(listener, args);
      }
      catch (e) { }
    });
  },

  /**
   * Notifies all providers that an add-on has been enabled when that type of
   * add-on only supports a single add-on being enabled at a time. This allows
   * the providers to disable theirs if necessary.
   * @param   id
   *          The id of the enabled add-on
   * @param   type
   *          The type of the enabled add-on
   * @param   pendingRestart
   *          A boolean indicating if the change will only take place the next
   *          time the application is restarted
   * @returns The nsIRDFContainer, initialized at the root.
   */
  notifyAddonSelected: function AM_notifyAddonSelected(id, type, pendingRestart) {
    this.providers.forEach(function(provider) {
      callProvider(provider, "addonSelected", null, id, type, pendingRestart);
    });
  },

  /**
   * Gets an AddonInstall for a URL.
   * @param   url
   *          The url the add-on is located at
   * @param   callback
   *          A callback to pass the AddonInstall to
   * @param   mimetype
   *          The mimetype of the add-on
   * @param   hash
   *          An optional hash of the add-on
   * @param   name
   *          An optional placeholder name while the add-on is being downloaded
   * @param   iconURL
   *          An optional placeholder icon URL while the add-on is being downloaded
   * @param   version
   *          An optional placeholder version while the add-on is being downloaded
   * @param   loadgroup
   *          An optional nsILoadGroup to associate any network requests with
   */
  getInstallForURL: function AM_getInstallForURL(url, callback, mimetype, hash, name, iconURL, version, loadgroup) {
    if (!url || !mimetype || !callback)
      throw new TypeError("Invalid arguments");

    for (let i = 0; i < this.providers.length; i++) {
      if (callProvider(this.providers[i], "supportsMimetype", false, mimetype)) {
        callProvider(this.providers[i], "getInstallForURL", null,
                     url, hash, name, iconURL, version, loadgroup, function(install) {
          safeCall(callback, install);
        });
        return;
      }
    }
    safeCall(callback, null);
  },

  /**
   * Gets an AddonInstall for an nsIFile.
   * @param   file
   *          the file the add-on is located at
   * @param   callback
   *          A callback to pass the AddonInstall to
   * @param   mimetype
   *          An optional mimetype hint for the add-on
   */
  getInstallForFile: function AM_getInstallForFile(file, callback, mimetype) {
    if (!file || !callback)
      throw new TypeError("Invalid arguments");

    // Freeze the list of providers then call each in turn till one gives a
    // valid Install
    let providers = this.providers.slice(0);

    function callNextProvider() {
      if (providers.length == 0)
        return safeCall(callback, null);

      let provider = providers.shift();
      if ("getInstallForFile" in provider) {
        callProvider(provider, "getInstallForFile", null, file, function(install) {
          if (install)
            safeCall(callback, install);
          else
            callNextProvider();
        });
      }
      else {
        callNextProvider();
      }
    }

    callNextProvider();
  },

  /**
   * Gets all current AddonInstalls optionally limiting to a set of types.
   * @param   types
   *          An optional array of types to retrieve
   * @param   callback
   *          A callback which will be passed an array of AddonInstalls
   */
  getInstalls: function AM_getInstalls(types, callback) {
    if (!callback)
      throw new TypeError("Invalid arguments");

    // Freeze the list of providers then call each in turn to get their Installs
    let providers = this.providers.slice(0);
    let installs = [];

    function callNextProvider() {
      if (providers.length == 0)
        return safeCall(callback, installs);

      let provider = providers.shift();
      if ("getInstalls" in provider) {
        callProvider(provider, "getInstalls", null, types, function(providerInstalls) {
          installs = installs.concat(providerInstalls);
          callNextProvider();
        });
      }
      else {
        callNextProvider()
      }
    }

    callNextProvider();
  },

  /**
   * Checks whether installation is enabled for a particular mimetype.
   * @param   mimetype
   *          The mimetype to check
   * @returns true if installation is enabled for the mimetype
   */
  isInstallEnabled: function AM_isInstallEnabled(mimetype) {
    for (let i = 0; i < this.providers.length; i++) {
      if (callProvider(this.providers[i], "supportsMimetype", false, mimetype))
        return callProvider(this.providers[i], "isInstallEnabled");
    }
    return false;
  },

  /**
   * Checkes whether a particular source is allowed to install add-ons of a
   * given mimetype.
   * @param   mimetype
   *          The mimetype of the add-on
   * @param   uri
   *          The uri of the source, may be null
   * @returns true if the source is allowed to install these types of add-on
   */
  isInstallAllowed: function AM_isInstallAllowed(mimetype, uri) {
    for (let i = 0; i < this.providers.length; i++) {
      if (callProvider(this.providers[i], "supportsMimetype", false, mimetype))
        return callProvider(this.providers[i], "isInstallAllowed", null, uri);
    }
  },

  /**
   * Starts installation of a set of AddonInstalls notifying the registered
   * web install listener of blocked or started installs.
   * @param   datasource
   *          The datasource the container is in
   * @param   root
   *          The RDF Resource which is the root of the container.
   * @returns The nsIRDFContainer, initialized at the root.
   */
  startInstallation: function AM_startInstallation(mimetype, source, uri, installs) {
    if (!("@mozilla.org/extensions/web-install-listener;1" in Cc)) {
      WARN("No web installer available, cancelling all installs");
      installs.forEach(function(install) {
        install.cancel();
      });
      return;
    }

    try {
      let weblistener = Cc["@mozilla.org/extensions/web-install-listener;1"].
                        getService(Ci.extIWebInstallListener);

      if (!this.isInstallAllowed(mimetype, uri)) {
        if (weblistener.onWebInstallBlocked(source, uri, installs, installs.length)) {
          installs.forEach(function(install) {
            install.install();
          });
        }
      }
      else if (weblistener.onWebInstallRequested(source, uri, installs, installs.length)) {
        installs.forEach(function(install) {
          install.install();
        });
      }
    }
    catch (e) {
      WARN("Failure calling web installer: " + e);
      installs.forEach(function(install) {
        install.cancel();
      });
      return;
    }
  },

  /**
   * Adds a new InstallListener.
   * @param   listener
   *          The InstallListener to add
   */
  addInstallListener: function AM_addInstallListener(listener) {
    this.removeInstallListener(listener);
    this.installListeners.push(listener);
  },

  /**
   * Removes an InstallListener.
   * @param   listener
   *          The InstallListener to remove
   */
  removeInstallListener: function AM_removeInstallListener(listener) {
    this.installListeners = this.installListeners.filter(function(i) {
      return i != listener;
    });
  },

  /**
   * Gets an add-on with a specific ID.
   * @param   id
   *          The ID of the add-on to retrieve
   * @param   callback
   *          The callback to pass the retrieved add-on to
   */
  getAddon: function AM_getAddon(id, callback) {
    if (!id || !callback)
      throw new TypeError("Invalid arguments");

    // Freeze the list of providers then call each in turn till one gives a
    // valid Install
    let providers = this.providers.slice(0);

    function callNextProvider() {
      if (providers.length == 0)
        return safeCall(callback, null);

      try {
        providers.shift().getAddon(id, function(addon) {
          if (addon)
            safeCall(callback, addon);
          else
            callNextProvider();
        });
      }
      catch (e) {
        ERROR("Exception calling provider.getAddon: " + e);
        callNextProvider();
      }
    }

    callNextProvider();
  },

  /**
   * Gets a set of add-ons.
   * @param   ids
   *          An array of IDs to retrieve
   * @param   callback
   *          The callback to pass an array of Addons to
   */
  getAddons: function AM_getAddon(ids, callback) {
    if (!ids || !callback)
      throw new TypeError("Invalid arguments");

    // Clone the array to avoid breaking the caller
    ids = ids.slice(0);
    let addons = [];

    function getNextAddon() {
      if (ids.length == 0)
        return safeCall(callback, addons);

      AddonManagerPrivate.getAddon(ids.shift(), function(addon) {
        addons.push(addon);
        getNextAddon();
      });
    }

    getNextAddon();
  },

  /**
   * Gets add-ons of specific types.
   * @param   types
   *          An array of types to retrieve or null to retrieve all types
   * @param   callback
   *          The callback to pass an array of Addons to.
   */
  getAddonsByType: function AM_getAddonsByType(types, callback) {
    if (!callback)
      throw new TypeError("Invalid arguments");

    // Freeze the list of providers then call each in turn to get their Installs
    let providers = this.providers.slice(0);
    let addons = [];

    function callNextProvider() {
      if (providers.length == 0)
        return safeCall(callback, addons);

      try {
        providers.shift().getAddonsByType(types, function(providerAddons) {
          addons = addons.concat(providerAddons);
          callNextProvider();
        });
      }
      catch (e) {
        ERROR("Exception calling provider.getAddonsByType: " + e);
        callNextProvider();
      }
    }

    callNextProvider();
  },

  /**
   * Gets add-ons that have operations waiting for an application restart to
   * take effect.
   * @param   types
   *          An optional array of types to limit the list to
   * @param   callback
   *          The callback to pass the array of Addons to
   */
  getAddonsWithPendingOperations: function AM_getAddonsWithPendingOperations(types, callback) {
    if (!callback)
      throw new TypeError("Invalid arguments");

    // Freeze the list of providers then call each in turn to get their Installs
    let providers = this.providers.slice(0);
    let addons = [];

    function callNextProvider() {
      if (providers.length == 0)
        return safeCall(callback, addons);

      let provider = providers.shift();
      if ("getAddonsWithPendingOperations" in provider) {
        callProvider(provider, "getAddonsWithPendingOperations", null,
                     types, function(providerAddons) {
          addons = addons.concat(providerAddons);
          callNextProvider();
        });
      }
      else {
        callNextProvider();
      }
    }

    callNextProvider();
  },

  /**
   * Adds a new AddonListener.
   * @param   listener
   *          The listener to add
   */
  addAddonListener: function AM_addAddonListener(listener) {
    this.removeAddonListener(listener);
    this.addonListeners.push(listener);
  },

  /**
   * Removes an AddonListener
   * @param   listener
   *          The listener to remove
   */
  removeAddonListener: function AM_removeAddonListener(listener) {
    this.addonListeners = this.addonListeners.filter(function(i) {
      return i != listener;
    });
  }
};

/**
 * This is a private API for the startup and platform integration code to use
 * It is not properly documented and subject to change at any point without
 * warning. Should not be used outside of core Mozilla code. All methods just
 * forward to AddonManagerPrivate.
 */
var AddonManagerInternal = {
  startup: function() {
    AddonManagerPrivate.startup();
  },

  registerProvider: function(provider) {
    AddonManagerPrivate.registerProvider(provider);
  },

  shutdown: function() {
    AddonManagerPrivate.shutdown();
  },

  backgroundUpdateCheck: function() {
    AddonManagerPrivate.backgroundUpdateCheck();
  },

  notifyAddonSelected: function(id, type, pendingRestart) {
    AddonManagerPrivate.notifyAddonSelected(id, type, pendingRestart);
  },

  callInstallListeners: function(method) {
    return AddonManagerPrivate.callInstallListeners.apply(AddonManagerPrivate, arguments);
  },

  callAddonListeners: function(method) {
    AddonManagerPrivate.callAddonListeners.apply(AddonManagerPrivate, arguments);
  }
};

/**
 * This is the public API that UI and developers should be calling. All methods
 * just forward to AddonManagerPrivate.
 */
var AddonManager = {
  STATE_AVAILABLE: 0,
  STATE_DOWNLOADING: 1,
  STATE_CHECKING: 2,
  STATE_DOWNLOADED: 3,
  STATE_DOWNLOAD_FAILED: 4,
  STATE_INSTALLING: 5,
  STATE_INSTALLED: 6,
  STATE_INSTALL_FAILED: 7,
  STATE_CANCELLED: 8,

  INSTALLERROR_INVALID_VERSION: -1,
  INSTALLERROR_INVALID_GUID: -2,
  INSTALLERROR_INCOMPATIBLE_VERSION: -3,
  INSTALLERROR_INCOMPATIBLE_PLATFORM: -5,
  INSTALLERROR_BLOCKLISTED: -6,
  INSTALLERROR_INSECURE_UPDATE: -7,
  INSTALLERROR_INVALID_MANIFEST: -8,
  INSTALLERROR_RESTRICTED: -9,
  INSTALLERROR_SOFTBLOCKED: -10,

  DOWNLOADERROR_NETWORK: -1,
  DOWNLOADERROR_INCORRECT_HASH: -2,
  DOWNLOADERROR_CORRUPT_FILE: -3,

  UPDATE_WHEN_USER_REQUESTED: 1,
  UPDATE_WHEN_NEW_APP_DETECTED: 2,
  UPDATE_WHEN_NEW_APP_INSTALLED: 3,
  UPDATE_WHEN_PERIODIC_UPDATE: 16,
  UPDATE_WHEN_ADDON_INSTALLED: 17,

  PENDING_ENABLE: 1,
  PENDING_DISABLE: 2,
  PENDING_UNINSTALL: 4,
  PENDING_INSTALL: 8,

  PERM_CANUNINSTALL: 1,
  PERM_CANENABLE: 2,
  PERM_CANDISABLE: 4,
  PERM_CANUPGRADE: 8,

  getInstallForURL: function(url, callback, mimetype, hash, name, iconURL, version, loadgroup) {
    AddonManagerPrivate.getInstallForURL(url, callback, mimetype, hash, name, iconURL, version, loadgroup);
  },

  getInstallForFile: function(file, callback, mimetype) {
    AddonManagerPrivate.getInstallForFile(file, callback, mimetype);
  },

  getAddon: function(id, callback) {
    AddonManagerPrivate.getAddon(id, callback);
  },

  getAddons: function(ids, callback) {
    AddonManagerPrivate.getAddons(ids, callback);
  },

  getAddonsWithPendingOperations: function(types, callback) {
    AddonManagerPrivate.getAddonsWithPendingOperations(types, callback);
  },

  getAddonsByType: function(types, callback) {
    AddonManagerPrivate.getAddonsByType(types, callback);
  },

  getInstalls: function(types, callback) {
    AddonManagerPrivate.getInstalls(types, callback);
  },

  isInstallEnabled: function(type) {
    return AddonManagerPrivate.isInstallEnabled(type);
  },

  isInstallAllowed: function(type, uri) {
    return AddonManagerPrivate.isInstallAllowed(type, uri);
  },

  startInstallation: function(type, source, uri, installs) {
    AddonManagerPrivate.startInstallation(type, source, uri, installs);
  },

  addInstallListener: function(listener) {
    AddonManagerPrivate.addInstallListener(listener);
  },

  removeInstallListener: function(listener) {
    AddonManagerPrivate.removeInstallListener(listener);
  },

  addAddonListener: function(listener) {
    AddonManagerPrivate.addAddonListener(listener);
  },

  removeAddonListener: function(listener) {
    AddonManagerPrivate.removeAddonListener(listener);
  }
};
