/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Extension Manager UI.
 *
 * The Initial Developer of the Original Code is
 * the Mozilla Foundation.
 * Portions created by the Initial Developer are Copyright (C) 2010
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Blair McBride <bmcbride@mozilla.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

const Cc = Components.classes;
const Ci = Components.interfaces;
const Cu = Components.utils;


Cu.import("resource://gre/modules/XPCOMUtils.jsm");
Cu.import("resource://gre/modules/Services.jsm");
Cu.import("resource://gre/modules/PluralForm.jsm");
Cu.import("resource://gre/modules/DownloadUtils.jsm");
Cu.import("resource://gre/modules/AddonManager.jsm");

// XXXunf this needs to come from a pref (and actually be implemented on AMO...)
// XXXunf also needs variables replaced - %locale%, %version% etc
const AMO_URL = "https://addons.mozilla.org/";


var gStrings = {};
XPCOMUtils.defineLazyServiceGetter(gStrings, "bundleSvc",
                                   "@mozilla.org/intl/stringbundle;1",
                                   "nsIStringBundleService");

XPCOMUtils.defineLazyGetter(gStrings, "brand", function() {
  return this.bundleSvc.createBundle("chrome://branding/locale/brand.properties");
});
XPCOMUtils.defineLazyGetter(gStrings, "ext", function() {
  return this.bundleSvc.createBundle("chrome://mozapps/locale/extensions/extensions.properties");
});
XPCOMUtils.defineLazyGetter(gStrings, "dl", function() {
  return this.bundleSvc.createBundle("chrome://mozapps/locale/downloads/downloads.properties");
});

XPCOMUtils.defineLazyGetter(gStrings, "brandShortName", function() {
  return this.brand.GetStringFromName("brandShortName");
});
XPCOMUtils.defineLazyGetter(gStrings, "appVersion", function() {
  return Services.appinfo.version;
});


var gTypeNames = {
  "recommended"   : "Recommended",
  "language"      : "Languages",
  "searchengine"  : "Search Engines",
  "extension"     : "Features",
  "theme"         : "Appearance",
  "plugin"        : "Plugins"
};


window.addEventListener("load",  initialize, false);
window.addEventListener("unload",  shutdown, false);

function initialize() {
  gCategories.initialize();
  gHeader.initialize();
  gViewController.initialize();
  gEventManager.initialize();

  gViewController.loadView("moz-em://list/extension");
}

function shutdown() {
  gEventManager.shutdown();
}

var gEventManager = {
  _listeners: {},
  _installListeners: [],

  initialize: function() {
    var self = this;
    ["onEnabling", "onDisabling", "onUninstalling", "onInstalling",
     "onOperationCancelled", "onUpdateAvailable",
     "onUpdateFinished"].forEach(function(aEvent) {
      self[aEvent] = function() {
        self.delegateAddonEvent(aEvent, [].splice.call(arguments, 0));
      };
    });

    ["onNewInstall", "onDownloadStarted", "onDownloadEnded", "onDownloadFailed",
     "onDownloadProgress", "onInstallStarted", "onInstallEnded",
     "onInstallFailed"].forEach(function(aEvent) {
      self[aEvent] = function() {
        self.delegateInstallEvent(aEvent, [].splice.call(arguments, 0));
      };
    });
    AddonManager.addInstallListener(this);
    AddonManager.addAddonListener(this);
  },

  shutdown: function() {
    AddonManager.removeInstallListener(this);
    AddonManager.removeAddonListener(this);
  },

  register: function(aListener, aAddonId) {
    if (!(aAddonId in this._listeners))
      this._listeners[aAddonId] = [];
    else if (this._listeners[aAddonId].indexOf(aListener) != -1)
      return;
    this._listeners[aAddonId].push(aListener);
  },

  unregister: function(aListener, aAddonId) {
    if (!(aAddonId in this._listeners))
      return;
    var index = this._listeners[aAddonId].indexOf(aListener);
    if (index == -1)
      return;
    this._listeners[aAddonId].splice(index, 1);
  },

  registerForInstalls: function(aListener) {
    if (this._installListeners.indexOf(aListener) != -1)
      return;
    this._installListeners.push(aListener);
  },

  unregisterForInstalls: function(aListener) {
    var i = this._installListeners.indexOf(aListener);
    if (i == -1)
      return;
    this._installListeners.splice(i, 1);
  },

  delegateAddonEvent: function(aEvent, aParams) {
    var addon = aParams.shift();
    if (!(addon.id in this._listeners))
      return;

    var listeners = this._listeners[addon.id];
    for (let i = 0; i < listeners.length; i++) {
      let listener = listeners[i];
      if (!(aEvent in listener))
        continue;
      try {
        listener[aEvent].apply(listener, aParams);
      } catch(e) {
        // this shouldn't be fatal
        Cu.reportError(e);
      }
    }
  },

  delegateInstallEvent: function(aEvent, aParams) {
    var install = aParams[0];
    if (install.existingAddon) {
      // install is an update
      let addon = install.existingAddon;
      this.delegateAddonEvent(aEvent, [addon].concat(aParams));
      return;
    }

    for (let i = 0; i < this._installListeners.length; i++) {
      let listener = this._installListeners[i];
      if (!(aEvent in listener))
        continue;
      try {
        listener[aEvent].apply(listener, aParams);
      } catch(e) {
        // this shouldn't be fatal
        Cu.reportError(e);
      }
    }
  }
};


var gViewController = {
  currentViewId: "",
  currentViewObj: null,
  previousViewId: "",
  viewObjects: {},

  initialize: function() {
    this.viewObjects["search"] = gSearchView;
    this.viewObjects["discover"] = gDiscoverView;
    this.viewObjects["list"] = gListView;
    this.viewObjects["detail"] = gDetailView;

    for each (let view in this.viewObjects)
      view.initialize();

    window.controllers.appendController(this);
  },

  parseViewId: function(aViewId) {
    var [,viewType, viewParam] = aViewId.match(/^moz-em:\/\/([^\/]+)\/(.*)$/) || [];
    return {type: viewType, param: viewParam};
  },

  loadView: function(aViewId) {
    if (aViewId == this.currentViewId)
      return;

    var view = this.parseViewId(aViewId);

    if (!view.type || !(view.type in this.viewObjects)) {
      /// XXXunf remove this warning
      Components.utils.reportError("Invalid view: " + view.type);
      return;
    }

    var viewObj = this.viewObjects[view.type];
    if (!viewObj.node) {
      /// XXXunf remove this warning
      Components.utils.reportError("Root node doesn't exist for '" + view.type + "' view");
      return;
    }

    if (this.currentViewObj) {
      try {
        let canHide = this.currentViewObj.hide();
        if (canHide === false)
          return;
      } catch (e) {
        // this shouldn't be fatal
        // XXXunf remove this warning
        Cu.reportError(e);
      }
      this.currentViewObj.node.hidden = true;
    }

    gCategories.select(aViewId);

    this.previousViewId = this.currentViewId;

    this.currentViewId = aViewId;
    this.currentViewObj = viewObj;

    this.currentViewObj.node.hidden = false;
    this.currentViewObj.show(view.param);
  },

  commands: {
    cmd_restartApp: {
      isEnabled: function() true,
      doCommand: function() {
        // Notify all windows that an application quit has been requested.
        var cancelQuit = Cc["@mozilla.org/supports-PRBool;1"]
                           .createInstance(Ci.nsISupportsPRBool);
        Services.obs.notifyObservers(cancelQuit, "quit-application-requested", "restart");

        // Something aborted the quit process.
        if (cancelQuit.data)
          return;

        Cc["@mozilla.org/toolkit/app-startup;1"]
          .getService(Ci.nsIAppStartup)
          .quit(Ci.nsIAppStartup.eRestart | Ci.nsIAppStartup.eAttemptQuit);
      }
    },

    cmd_showItemDetails: {
      isEnabled: function(aAddon) {
        return !!aAddon;
      },
      doCommand: function(aAddon) {
        gViewController.loadView("moz-em://detail/" + aAddon.id);
      }
    },

    cmd_findAllUpdates: {
      isEnabled: function() true,
      doComand: function() {
        //XXXunf hook this up so the relevant UI gets events
        var listener = {
          onUpdateAvailable: function(aAddon, aInstall) {
            gEventManager.delegateAddonEvent("onUpdateAvailable", [aAddon, aInstall]);
            aInstall.install();
          },
          onUpdateFinished: function(aAddon, aError) {
            gEventManager.delegateAddonEvent("onUpdateFinished", [aAddon, aError]);
          }
        };
        AddonManager.getAddonsByType(null, function(aAddonList) {
          aAddonList.forEach(function(aAddon) {
            if (aAddon.permissions & AddonManager.PERM_CANUPGRADE) {
              aAddon.findUpdates(listener,
                                 AddonManager.UPDATE_WHEN_USER_REQUESTED);
            }
          });
        });
      }
    },

    cmd_findItemUpdates: {
      isEnabled: function(aAddon) {
        if (!aAddon)
          return false;
        return hasPermission(aAddon, "upgrade");
      },
      doCommand: function(aAddon) {
        //XXXunf hook this up so the relevant UI gets events
        var listener = {
          onUpdateAvailable: function(aAddon, aInstall) {
            gEventManager.delegateAddonEvent("onUpdateAvailable", [aAddon, aInstall]);
            aInstall.install();
          },
          onUpdateFinished: function(aAddon, aError) {
            gEventManager.delegateAddonEvent("onUpdateFinished", [aAddon, aError]);
          }
        };
        gEventManager.delegateAddonEvent("onCheckingUpdate", [aAddon]);
        aAddon.findUpdates(listener, AddonManager.UPDATE_WHEN_USER_REQUESTED);
      }
    },

    cmd_showItemPreferences: {
      isEnabled: function(aAddon) {
        if (!aAddon)
          return false;
        return !!aAddon.optionsURL;
      },
      doCommand: function(aAddon) {
        var optionsURL = aAddon.optionsURL;
        var windows = Services.wm.getEnumerator(null);
        while (windows.hasMoreElements()) {
          var win = windows.getNext();
          if (win.document.documentURI == optionsURL) {
            win.focus();
            return;
          }
        }
        var features = "chrome,titlebar,toolbar,centerscreen";
        try {
          var instantApply = Services.prefs.getBoolPref("browser.preferences.instantApply");
          features += instantApply ? ",dialog=no" : ",modal";
        } catch (e) {
          features += ",modal";
        }
        openDialog(optionsURL, "", features);
      }
    },

    cmd_showItemAbout: {
      isEnabled: function(aAddon) {
        if (!aAddon)
          return false;
        // XXXunf check to make sure this is an installed item
        return true;
      },
      doCommand: function(aAddon) {
        var aboutURL = aAddon.aboutURL;
        if (aboutURL)
          openDialog(aboutURL, "", "chrome,centerscreen,modal");
        else
          openDialog("chrome://mozapps/content/extensions/about.xul",
                     "", "chrome,centerscreen,modal", aAddon);
      }
    },

    cmd_enableItem: {
      isEnabled: function(aAddon) {
        if (!aAddon)
          return false;
        return hasPermission(aAddon, "enable");
      },
      doCommand: function(aAddon) {
        aAddon.userDisabled = false;
      }
    },

    cmd_disableItem: {
      isEnabled: function(aAddon) {
        if (!aAddon)
          return false;
        return hasPermission(aAddon, "disable");
      },
      doCommand: function(aAddon) {
        aAddon.userDisabled = true;
      }
    },

    cmd_uninstallItem: {
      isEnabled: function(aAddon) {
        if (!aAddon)
          return false;
        return hasPermission(aAddon, "uninstall");
      },
      doCommand: function(aAddon) {
        aAddon.uninstall();
      }
    },

    cmd_cancelUninstallItem: {
      isEnabled: function(aAddon) {
        if (!aAddon)
          return false;
        return isPending(aAddon, "uninstall");
      },
      doCommand: function(aAddon) {
        aAddon.cancelUninstall();
      }
    }
  },

  supportsCommand: function(aCommand) {
    if (aCommand in this.commands)
      return true;
    return false;
  },

  isCommandEnabled: function(aCommand) {
    var addon = this.currentViewObj.getSelectedAddon();
    return this.commands[aCommand].isEnabled(addon);
  },

  updateCommands: function() {
    // wait until the view is initialized
    if (!this.currentViewObj)
      return;
    var addon = this.currentViewObj.getSelectedAddon();
    for (let commandId in this.commands) {
      let cmd = document.getElementById(commandId);
      cmd.setAttribute("disabled", !this.commands[commandId].isEnabled(addon));
    }
  },

  doCommand: function(aCommand) {
    var addon = this.currentViewObj.getSelectedAddon();
    var cmd = this.commands[aCommand];
    if (!cmd.isEnabled(addon))
      return;
    cmd.doCommand(addon);
  },

  onEvent: function() {}
};


function prettifyURL(aURL) {
  if (!aURL)
    return "";
  return aURL.replace(/^http(s|)\:\/\//, "")
             .replace(/^www./, "")
             .replace(/\/$/, "");
}


function $(aNodeId) {
  return document.getElementById(aNodeId);
}


function hasPermission(aAddon, aPerm) {
  var perm = AddonManager["PERM_CAN" + aPerm.toUpperCase()];
  return !!(aAddon.permissions & perm);
}


function isPending(aAddon, aAction) {
  var action = AddonManager["PENDING_" + aAction.toUpperCase()];
  return !!(aAddon.pendingOperations & action);
}


function createItem(aAddon, aIsInstall) {
  let item = document.createElement("richlistitem");

  item.setAttribute("name", aAddon.name);
  item.setAttribute("type", aAddon.type);

  if (aIsInstall) {
    item.mInstall = aAddon;
    item.setAttribute("status", "installing");
  } else {
    item.mAddon = aAddon;

    if (isPending(aAddon, "uninstall"))
      item.setAttribute("status", "uninstalled");
    else
      item.setAttribute("status", "installed");

    // set only attributes needed for sorting and XBL binding,
    // the binding handles the rest
    item.setAttribute("value", aAddon.id);
    item.setAttribute("dateUpdated", (new Date).valueOf() - Math.floor(Math.random() * 100000000000)); // XXXapi
    var size = Math.floor(Math.random() * 1024 * 1024 * 2);
    size = ("00000000000" + size).slice(-10); // HACK: nsIXULSortService doesn't do numerical sorting
    item.setAttribute("size", size); // XXXapi
  }
  return item;
}


var gCategories = {
  node: null,
  _search: null,

  initialize: function() {
    this.node = document.getElementById("categories");
    this._search = this.get("moz-em://search/");

    this.maybeHideSearch();

    var self = this;
    this.node.addEventListener("select", function() {
      self.maybeHideSearch();
      gViewController.loadView(self.node.selectedItem.value);
    }, false);
  },

  select: function(aId) {
    if (this.node.selectedItem &&
        this.node.selectedItem.value == aId)
      return;

    var view = gViewController.parseViewId(aId);
    if (view.type == "detail")
      return;

    if (view.type == "search")
      aId = "moz-em://search/";

    var item = this.get(aId);
    if (item) {
      item.hidden = false;
      this.node.suppressOnSelect = true;
      this.node.selectedItem = item;
      this.node.suppressOnSelect = false;
      this.node.ensureElementIsVisible(item);

      this.maybeHideSearch();
    }
  },

  get: function(aId) {
    for (let i = 0; i < this.node.itemCount; i++) {
      let item = this.node.getItemAtIndex(i);
      if (item.value == aId)
        return item;
    }
    return null;
  },

  setBadge: function(aId, aCount) {
    let item = this.get(aId);
    if (item)
      item.badgeCount = aCount;
  },

  maybeHideSearch: function() {
    var view = gViewController.parseViewId(this.node.selectedItem.value);
    this._search.hidden = view.type != "search";
  }
};


var gHeader = {
  _search: null,
  _name: null,
  _linkText: null,
  _linkContainer: null,
  _dest: "",

  initialize: function() {
    this._name = document.getElementById("header-name");
    this._linkContainer = document.getElementById("header-link-container");
    this._linkText = document.getElementById("header-link");
    this._search = document.getElementById("header-search");

    var self = this;
    this._linkText.addEventListener("click", function() {
      gViewController.loadView(self._dest);
    }, false);

    this._search.addEventListener("command", function(aEvent) {
      var query = aEvent.target.value;
      gViewController.loadView("moz-em://search/" + query);
    }, false);

    this.setName("");
  },

  setName: function(aName) {
    this._name.value = aName;
    this._name.hidden = false;
    this._linkContainer.hidden = true;
  },

  showBackButton: function() {
    this._linkText.value = "Back to " + this._name.value;
    this._dest = gViewController.previousViewId;
    this._name.hidden = true;
    this._linkContainer.hidden = false;
  },

  get searchQuery() {
    return this._search.value;
  },

  set searchQuery(aQuery) {
    this._search.value = aQuery;
  }
};


var gDiscoverView = {
  node: null,
  _browser: null,
  initialize: function() {
    this.node = document.getElementById("discover-view");
    this._browser = document.getElementById("discover-browser");
    this._browser.homePage = AMO_URL;
  },
  show: function() {
    gHeader.setName("Discover Add-ons");
    // load content only if we're not already showing something on AMO
    // XXXunf should only be comparing hostname
    if (this._browser.currentURI.spec.indexOf(this._browser.homePage) == -1)
      this._browser.goHome();

    gViewController.updateCommands();
  },
  hide: function() { },
  getSelectedAddon: function() null
};


var gSearchView = {
  node: null,
  _sorters: null,
  _listBox: null,

  initialize: function() {
    this.node = document.getElementById("search-view");
    this._sorters = document.getElementById("search-sorters");
    this._sorters.handler = this;
    this._listBox = document.getElementById("search-list");
  },

  show: function(aQuery) {
    gHeader.setName("Search");

    gHeader.searchQuery = aQuery;
    aQuery = aQuery.trim().toLocaleLowerCase();

    while (this._listBox.itemCount > 0)
      this._listBox.removeItemAt(0);

    var self = this;
    AddonManager.getAddonsByType([], function(aAddonsList) {
      for (let i = 0; i < aAddonsList.length; i++) {
        let addon = aAddonsList[i];
        let score = 0;
        if (aQuery.length > 0) {
          score = self.getMatchScore(addon, aQuery);
          if (score == 0)
            continue;
        }
        let item = createItem(addon);
        item.setAttribute("relevancescore", score);
        self._listBox.appendChild(item);
      }

      self.onSortChanged("relevancescore", false);
      gViewController.updateCommands();
    });
  },

  hide: function() { },

  getMatchScore: function(aAddon, aQuery) {
    var score = 0;
    score += this.calculateMatchScore(aAddon.name, aQuery, 2);
    score += this.calculateMatchScore(aAddon.description, aQuery, 1);
    return score;
  },

  calculateMatchScore: function(aStr, aQuery, aMultiplier) {
    var score = 0;
    if (aQuery.length == 0)
      return score;

    aStr = aStr.trim().toLocaleLowerCase();
    var haystack = aStr.split(/\W+/);
    var needles = aQuery.split(/\W+/);

    for (let n = 0; n < needles.length; n++) {
      for (let h = 0; h < haystack.length; h++) {
        if (haystack[h] == needles[n]) {
          // matching whole words is best
          score += 1;
        } else {
          let i = haystack[h].indexOf(needles[n]);
          if (i == 0) // matching on word boundries is also good
            score += 0.6;
          else if (i > 0) // substring matches not so good
            score += 0.3;
        }
      }
    }

    // give progressively higher score for longer queries, since longer queries
    // are more likely to be unique and therefore more relevant.
    if (needles.length > 1 && aStr.indexOf(aQuery) != -1)
      score += needles.length;

    return score * aMultiplier;
  },

  onSortChanged: function(aSortBy, aAscending) {
    var sortService = Cc["@mozilla.org/xul/xul-sort-service;1"]
                        .getService(Ci.nsIXULSortService);
    sortService.sort(this._listBox, aSortBy, aAscending ? "ascending" : "descending");
  },

  getSelectedAddon: function() {
    var item = this._listBox.selectedItem;
    if (item)
      return item.mAddon;
    return null;
  }

};


var gListView = {
  node: null,
  _listBox: null,
  _sorters: null,

  initialize: function() {
    this.node = document.getElementById("list-view");
    this._sorters = document.getElementById("list-sorters");
    this._sorters.handler = this;
    this._listBox = document.getElementById("addon-list");
  },
  show: function(aType) {
    gHeader.setName(gTypeNames[aType]);

    var types = [aType];
    if (aType == "extension")
      types.push("bootstrapped");

    while (this._listBox.itemCount > 0)
      this._listBox.removeItemAt(0);

    var self = this;
    AddonManager.getAddonsByType(types, function(aAddonsList) {
      window.__addons = aAddonsList; // XXXunf remove this!
      for (let i = 0; i < aAddonsList.length; i++) {
        let item = createItem(aAddonsList[i]);
        self._listBox.appendChild(item);
      }

      self.onSortChanged(self._sorters.sortBy, self._sorters.ascending);
      gViewController.updateCommands();

      AddonManager.getInstalls(null, function(aInstallsList) {
      window.__installs = aInstallsList; // XXXunf remove this!
        for (let i = 0; i < aInstallsList.length; i++) {
          let install = aInstallsList[i];
          // XXXunf attach upgrade installs to existing items
          if (install.existingAddon)
            continue;
          let item = createItem(install, true);
          self._listBox.insertBefore(item, self._listBox.firstChild);
        }
      });
    });

    gEventManager.registerForInstalls(this);
  },

  hide: function() {
    gEventManager.unregisterForInstalls(this);
  },

  onSortChanged: function(aSortBy, aAscending) {
    var sortService = Cc["@mozilla.org/xul/xul-sort-service;1"]
                        .getService(Ci.nsIXULSortService);
    sortService.sort(this._listBox, aSortBy, aAscending ? "ascending" : "descending");
  },

  onNewInstall: function(aInstall) {
    window.__install = aInstall; // XXXunf remove this!
    var item = createItem(aInstall, true);
    this._listBox.insertBefore(item, this._listBox.firstChild);
  },

  getSelectedAddon: function() {
    var item = this._listBox.selectedItem;
    if (item)
      return item.mAddon;
    return null;
  }
};


var gDetailView = {
  node: null,
  _addon: null,
  initialize: function() {
    this.node = document.getElementById("detail-view");
  },
  show: function(aAddonId) {
    var self = this;
    this.node.setAttribute("loading", true);
    setTimeout(function() {
      if (self.node.hasAttribute("loading"))
        self.node.setAttribute("loading-extended", true);
    }, 100);
    gHeader.showBackButton();

    AddonManager.getAddon(aAddonId, function(aAddon) {
      self._addon = aAddon;
      self.clearLoading();
      self.node.setAttribute("type", aAddon.type);

      $("detail-name").value = aAddon.name;
      $("detail-icon").src = aAddon.iconURL;
      $("detail-name").value = aAddon.name;
      $("detail-creator").setCreator(aAddon.creator, aAddon.creatorURL || aAddon.homepageURL);
      $("detail-creator2").setCreator(aAddon.creator, aAddon.creatorURL || aAddon.homepageURL);
      $("detail-homepage").value = prettifyURL(aAddon.homepageURL);
      $("detail-homepage").href = aAddon.homepageURL;
      $("detail-desc").value = aAddon.description;
      $("detail-version").value = aAddon.version;

      $("detail-autoupdate").checked = aAddon.updateAutomatically;
      var canUpdate = hasPermission(aAddon, "upgrade");
      $("detail-autoupdate").hidden = !canUpdate;
      $("detail-findupdates").hidden = !canUpdate;
      $("detail-prefs").hidden = !aAddon.optionsURL;

      gViewController.updateCommands();
    });
  },

  hide: function() {
    this.clearLoading();
  },

  clearLoading: function() {
    this.node.removeAttribute("loading");
    this.node.removeAttribute("loading-extended");
  },

  getSelectedAddon: function() {
    return this._addon;
  }
};
