// -*- Mode: js2; tab-width: 2; indent-tabs-mode: nil; js2-basic-offset: 2; js2-skip-preprocessor-directives: t; -*-
/*
 * ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Mozilla Mobile Browser.
 *
 * The Initial Developer of the Original Code is
 * Mozilla Corporation.
 * Portions created by the Initial Developer are Copyright (C) 2009
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Roy Frostig <rfrostig@mozilla.com>
 *   Ben Combee <bcombee@mozilla.com>
 *   Matt Brubeck <mbrubeck@mozilla.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

let Ci = Components.interfaces;

// Blindly copied from Safari documentation for now.
const kViewportMinScale  = 0;
const kViewportMaxScale  = 10;
const kViewportMinWidth  = 200;
const kViewportMaxWidth  = 10000;
const kViewportMinHeight = 223;
const kViewportMaxHeight = 10000;

// -----------------------------------------------------------
// General util/convenience tools
//

let Util = {
  bind: function bind(f, thisObj) {
    return function() {
      return f.apply(thisObj, arguments);
    };
  },

  bindAll: function bindAll(instance) {
    let bind = Util.bind;
    for (let key in instance)
      if (instance[key] instanceof Function)
        instance[key] = bind(instance[key], instance);
  },

  /** printf-like dump function */
  dumpf: function dumpf(str) {
    var args = arguments;
    var i = 1;
    dump(str.replace(/%s/g, function() {
      if (i >= args.length) {
        throw "dumps received too many placeholders and not enough arguments";
      }
      return args[i++].toString();
    }));
  },

  /** Like dump, but each arg is handled and there's an automatic newline */
  dumpLn: function dumpLn() {
    for (var i = 0; i < arguments.length; i++) { dump(arguments[i] + " "); }
    dump("\n");
  },

  getWindowUtils: function getWindowUtils(win) {
    return win.QueryInterface(Ci.nsIInterfaceRequestor).getInterface(Ci.nsIDOMWindowUtils);
  },

  getScrollOffset: function getScrollOffset(win) {
    var cwu = Util.getWindowUtils(win);
    var scrollX = {};
    var scrollY = {};
    cwu.getScrollXY(false, scrollX, scrollY);
    return new Point(scrollX.value, scrollY.value);
  },

  /** Executes aFunc after other events have been processed. */
  executeSoon: function executeSoon(aFunc) {
    let tm = Cc["@mozilla.org/thread-manager;1"].getService(Ci.nsIThreadManager);
    tm.mainThread.dispatch({
      run: function() {
        aFunc();
      }
    }, Ci.nsIThread.DISPATCH_NORMAL);
  },

  getHrefForElement: function getHrefForElement(target) {
    // XXX: This is kind of a hack to work around a Gecko bug (see bug 266932)
    // We're going to walk up the DOM looking for a parent link node.
    // This shouldn't be necessary, but we're matching the existing behaviour for left click

    let link = null;
    while (target) {
      if (target instanceof HTMLAnchorElement || 
          target instanceof HTMLAreaElement ||
          target instanceof HTMLLinkElement) {
          if (target.hasAttribute("href"))
            link = target;
      }
      target = target.parentNode;
    }

    if (link && link.hasAttribute("href"))
      return link.href;
    else
      return null;
  },

  makeURLAbsolute: function makeURLAbsolute(base, url) {
    // Note:  makeURI() will throw if url is not a valid URI
    return makeURI(url, null, makeURI(base)).spec;
  },

  getViewportMetadata: function getViewportMetadata(win) {
    let dpiScale = gPrefService.getIntPref("zoom.dpiScale") / 100;

    let doctype = win.document.doctype;
    if (doctype && /(WAP|WML|Mobile)/.test(doctype.publicId))
      return { defaultZoom: dpiScale, autoSize: true };

    let windowUtils = win.QueryInterface(Ci.nsIInterfaceRequestor)
                         .getInterface(Ci.nsIDOMWindowUtils);
    let handheldFriendly = windowUtils.getDocumentMetadata("HandheldFriendly");
    if (handheldFriendly == "true")
      return { defaultZoom: dpiScale, autoSize: true };

    if (win.document instanceof XULDocument)
      return { defaultZoom: 1.0, autoSize: true, allowZoom: false };

    // viewport details found here
    // http://developer.apple.com/safari/library/documentation/AppleApplications/Reference/SafariHTMLRef/Articles/MetaTags.html
    // http://developer.apple.com/safari/library/documentation/AppleApplications/Reference/SafariWebContent/UsingtheViewport/UsingtheViewport.html
    
    // Note: These values will be NaN if parseFloat or parseInt doesn't find a number.
    // Remember that NaN is contagious: Math.max(1, NaN) == Math.min(1, NaN) == NaN.
    let viewportScale = parseFloat(windowUtils.getDocumentMetadata("viewport-initial-scale"));
    let viewportMinScale = parseFloat(windowUtils.getDocumentMetadata("viewport-minimum-scale"));
    let viewportMaxScale = parseFloat(windowUtils.getDocumentMetadata("viewport-maximum-scale"));
    let viewportWidthStr = windowUtils.getDocumentMetadata("viewport-width");
    let viewportHeightStr = windowUtils.getDocumentMetadata("viewport-height");

    viewportScale = Util.clamp(viewportScale, kViewportMinScale, kViewportMaxScale);
    viewportMinScale = Util.clamp(viewportMinScale, kViewportMinScale, kViewportMaxScale);
    viewportMaxScale = Util.clamp(viewportMaxScale, kViewportMinScale, kViewportMaxScale);

    // If initial scale is 1.0 and width is not set, assume width=device-width
    let autoSize = (viewportWidthStr == "device-width" ||
                    viewportHeightStr == "device-height" ||
                    (viewportScale == 1.0 && !viewportWidthStr));

    let viewportWidth = Util.clamp(parseInt(viewportWidthStr), kViewportMinWidth, kViewportMaxWidth);
    let viewportHeight = Util.clamp(parseInt(viewportHeightStr), kViewportMinHeight, kViewportMaxHeight);

    // Zoom level is the final (device pixel : CSS pixel) ratio for content.
    // Since web content specifies scale as (reference pixel : CSS pixel) ratio,
    // multiply the requested scale by a constant (device pixel : reference pixel)
    // factor to account for high DPI devices.
    //
    // See bug 561445 or any of the examples of chrome/tests/browser_viewport_XX.html
    // for more information and examples.
    let defaultZoom = viewportScale * dpiScale;
    let minZoom = viewportMinScale * dpiScale;
    let maxZoom = viewportMaxScale * dpiScale;

    return {
      defaultZoom: defaultZoom,
      minZoom: minZoom,
      maxZoom: maxZoom,
      width: viewportWidth,
      height: viewportHeight,
      autoSize: autoSize,
      allowZoom: windowUtils.getDocumentMetadata("viewport-user-scalable") != "no"
    };
  },

  clamp: function(num, min, max) {
    return Math.max(min, Math.min(max, num));
  },

  /**
   * Determines whether a home page override is needed.
   * Returns:
   *  "new profile" if this is the first run with a new profile.
   *  "new version" if this is the first run with a build with a different
   *                      Gecko milestone (i.e. right after an upgrade).
   *  "none" otherwise.
   */
  needHomepageOverride: function needHomepageOverride() {
    let savedmstone = null;
    try {
      savedmstone = gPrefService.getCharPref("browser.startup.homepage_override.mstone");
    } catch (e) {}

    if (savedmstone == "ignore")
      return "none";

   let ourmstone = "2.0a1pre";

    if (ourmstone != savedmstone) {
      gPrefService.setCharPref("browser.startup.homepage_override.mstone", ourmstone);

      return (savedmstone ? "new version" : "new profile");
    }

    return "none";
  },

  /** Don't display anything in the urlbar for these special URIs. */
  isURLEmpty: function isURLEmpty(aURL) {
    return (!aURL || aURL == "about:blank" || aURL == "about:home");
  },

  /** Recursively find all documents, including root document. */
  getAllDocuments: function getAllDocuments(doc, resultSoFar) {
    resultSoFar = resultSoFar || [doc];
    if (!doc.defaultView)
      return resultSoFar;
    let frames = doc.defaultView.frames;
    if (!frames)
      return resultSoFar;

    let i;
    let currentDoc;
    for (i = 0; i < frames.length; i++) {
      currentDoc = frames[i].document;
      resultSoFar.push(currentDoc);
      this.getAllDocuments(currentDoc, resultSoFar);
    }

    return resultSoFar;
  },

  // Put the Mozilla networking code into a state that will kick the auto-connection
  // process.
  forceOnline: function forceOnline() {
//@line 260 "/home/cltbld/build/mobile-electrolysis-maemo5-qt-nightly/electrolysis/mobile/chrome/content/Util.js"
    gIOService.offline = false;
//@line 262 "/home/cltbld/build/mobile-electrolysis-maemo5-qt-nightly/electrolysis/mobile/chrome/content/Util.js"
  },

  /**
   * Put this function in your object for event handling.  This will call a function
   * specific to your event, e.g. "handleMousedown" or "handleDOMLinkAdded".
   */
  handleEvent: function(e) {
    let name = "handle" + Util.capitalize(e.type);
    let fn = this[name];
    return fn.apply(this, arguments);
  },

  /**
   * Similar to handleEvent, but for cross-process messages.
   */
  receiveMessage: function(message) {
    let name = "receive" + Util.capitalize(message.name);
    let fn = this[name] || this["receiveDefault"];
    if (!fn)
      throw "Received a message " + message.name + " but did not do anything about it!";

    let args = [message].concat(message.json);
    return { name: "Response", message: fn.apply(this, args) };
  },

  /** Capitalize first letter of a string. */
  capitalize: function(str) {
    return str.charAt(0).toUpperCase() + str.substring(1);
  },
  
  isPortrait: function isPortrait() {
    return (window.innerWidth < 500);
  }
};


/**
 * Helper class to nsITimer that adds a little more pizazz.  Callback can be an
 * object with a notify method or a function.
 */
Util.Timeout = function(callback) {
  this._callback = callback;
  this._timer = Cc["@mozilla.org/timer;1"].createInstance(Ci.nsITimer);
  this._active = false;
}

Util.Timeout.prototype = {
  /** Timer callback. Don't call this manually. */
  notify: function notify() {
    this._active = false;
    if (this._callback.notify)
      this._callback.notify();
    else
      this._callback.apply(null);
  },

  /** Do the callback once.  Cancels other timeouts on this object. */
  once: function once(delay, callback) {
    if (callback)
      this._callback = callback;
    this.clear();
    this._timer.initWithCallback(this, delay, this._timer.TYPE_ONE_SHOT);
    this._active = true;
    return this;
  },

  /** Do the callback every delay msecs. Cancels other timeouts on this object. */
  interval: function interval(delay, callback) {
    if (callback)
      this._callback = callback;
    this.clear();
    this._timer.initWithCallback(this, delay, this._timer.TYPE_REPEATING_SLACK);
    this._active = true;
    return this;
  },

  /** Clear any pending timeouts. */
  clear: function clear() {
    if (this._active) {
      this._timer.cancel();
      this._active = false;
    }
    return this;
  },

  /** If there is a pending timeout, call it and cancel the timeout. */
  flush: function flush() {
    if (this._active) {
      this.clear();
      this.notify();
    }
    return this;
  },

  /** Return true iff we are waiting for a callback. */
  isPending: function isPending() {
    return this._active;
  }
};


/** A form assistant wrapper for content elements. */
function BasicWrapper(element) {
  if (!element)
    throw "Instantiating BasicWrapper with null element";
  this.element = element;

  if (!this._fac)
    BasicWrapper.prototype._fac = Cc["@mozilla.org/satchel/form-autocomplete;1"].getService(Ci.nsIFormAutoComplete);
}

BasicWrapper.prototype = {
  isVisible: function isVisible() {
    return this._isElementVisible(this.element);
  },

  /** Can the next and previous buttons get to this element? Does not factor in visibility. */
  canNavigateTo: function canNavigateTo() {
    let element = this.element;
    if (element.disabled)
      return false;

    if (element.getAttribute("role") == "button" && element.hasAttribute("tabindex"))
      return true;

    if (this.hasChoices() || element instanceof HTMLTextAreaElement)
      return true;

    if (element instanceof HTMLInputElement || element instanceof HTMLButtonElement) {
      return !(element.type == "hidden")
    }

    return false;
  },

  /** Should assistant act when user taps on element? */
  canAssist: function canAssist() {
    let element = this.element;

    let formExceptions = {button: true, checkbox: true, file: true, image: true, radio: true, reset: true, submit: true};
    if (element instanceof HTMLInputElement && formExceptions[element.type])
      return false;

    if (element instanceof HTMLButtonElement ||
        (element.getAttribute("role") == "button" && element.hasAttribute("tabindex")))
      return false;

    return this.canNavigateTo();
  },

  /** Gets a rect bounding important parts of the element that must be seen when assisting. */
  getRect: function getRect() {
    const kDistanceMax = 100;
    let element = this.element;
    let elRect = getBoundingContentRect(element);

    let labels = this._getLabelsFor(element);
    for (let i=0; i<labels.length; i++) {
      let labelRect = getBoundingContentRect(labels[i]);
      if (labelRect.left < elRect.left) {
        let isClose = Math.abs(labelRect.left - elRect.left) - labelRect.width < kDistanceMax &&
                      Math.abs(labelRect.top - elRect.top) - labelRect.height < kDistanceMax;
        if (isClose) {
          let width = labelRect.width + elRect.width + (elRect.left - labelRect.left - labelRect.width);
          return new Rect(labelRect.left, labelRect.top, width, elRect.height).expandToIntegers();
        }
      }
    }
    return elRect;
  },

  /** Element is capable of having autocomplete suggestions. */
  canAutocomplete: function() {
    return this.element instanceof HTMLInputElement;
  },

  /** Caret is used to input text for this element. */
  getCaretRect: function() {
    let element = this.element;
    if ((element instanceof HTMLTextAreaElement ||
        (element instanceof HTMLInputElement && element.type == "text")) &&
        gFocusManager.focusedElement == element) {
      let utils = Util.getWindowUtils(element.ownerDocument.defaultView);
      let rect = utils.sendQueryContentEvent(utils.QUERY_CARET_RECT, element.selectionEnd, 0, 0, 0);
      if (!rect)
        return new Rect(0, 0, 0, 0);

      let scroll = Util.getScrollOffset(element.ownerDocument.defaultView);
      let caret = new Rect(scroll.x + rect.left, scroll.y + rect.top, rect.width, rect.height);
      return caret;
    }

    return new Rect(0, 0, 0, 0);
  },

  /** Create a list of suggestions for input autocomplete. Returns array of strings. */
  getAutocompleteSuggestions: function() {
    if (!this.canAutocomplete())
      return [];
    return [];

    // XXX
    // Autocomplete fetch does not work with input fields in content, which leads me to believe
    // setting autocomplete entries won't work when text is inputted into a box either.
    let element = this.element;
    let suggestions = [];
    let currentValue = element.value;
    let results = this._fac.autoCompleteSearch(element.name, currentValue, element, null);
    if (results.matchCount > 0) {
      for (let i = 0; i < results.matchCount; i++) {
        let value = results.getValueAt(i);
        suggestions.push(value);
      }
    }
    return suggestions;
  },

  autocomplete: function(value) {
    this.element.value = value;
  },

  /** Returns true iff the choices interface needs to shown. */
  hasChoices: function hasChoices() {
    let element = this.element;
    return (element instanceof HTMLSelectElement) ||
      (element instanceof Ci.nsIDOMXULMenuListElement);
  },

  choiceSelect: function(index, selected, clearAll) {
    let wrapper = this._getChoiceWrapper(this._currentIndex);
    if (wrapper)
      wrapper.select(index, selected, clearAll);
  },

  choiceChange: function() {
    let wrapper = this._getChoiceWrapper(this._currentIndex);
    if (wrapper)
      wrapper.fireOnChange();
  },

  getChoiceData: function() {
    let wrapper = this._getChoiceWrapper();
    if (!wrapper)
      return null;

    let optionIndex = 0;
    let result = {
      multiple: wrapper.getMultiple(),
      choices: []
    };
    
    // Build up a flat JSON array of the choices. In HTML, it's possible for select element choices
    // to be under a group header (but not recursively). We distinguish between headers and entries
    // using the boolean "choiceData.group".
    // XXX If possible, this would be a great candidate for tracing.
    let children = wrapper.getChildren();
    for (let i = 0; i < children.length; i++) {
      let child = children[i];
      if (wrapper.isGroup(child)) {
        // This is the group element. Add an entry in the choices that says that the following
        // elements are a member of this group.
        result.choices.push({ group: true, groupName: child.label });
        let subchildren = child.children;
        for (let ii = 0; i < subchildren.length; ii++) {
          let subchild = subchildren[ii];
          result.choices.push({
            group: false,
            inGroup: true,
            text: wrapper.getText(subchild),
            selected: subchild.selected,
            optionIndex: optionIndex++
          });
        }
      } else if (wrapper.isOption(child)) {
        // This is a regular choice under no group.
        result.choices.push({
          group: false,
          inGroup: false,
          text: wrapper.getText(child),
          selected: child.selected,
          optionIndex: optionIndex++
        });
      }
    }

    return result;
  },

  getJSON: function() {
    return {
      choiceData: this.getChoiceData(),
      canAutocomplete: this.canAutocomplete(),
      autocomplete: this.getAutocompleteSuggestions(),
      navigable: this.canNavigateTo(),
      assistable: this.canAssist(),
      rect: this.getRect()
    };
  },

  _getChoiceWrapper: function() {
    let choiceWrapper = null;
    let element = this.element;
    if (element instanceof HTMLSelectElement)
      choiceWrapper = new SelectWrapper(element);
    else if (element instanceof Ci.nsIDOMXULMenuListElement)
      choiceWrapper = new MenulistWrapper(element);
    return choiceWrapper;
  },

  _getLabelsFor: function(element) {
    let associatedLabels = [];

    let labels = element.ownerDocument.getElementsByTagName("label");
    for (let i=0; i<labels.length; i++) {
      if (labels[i].getAttribute("for") == element.id)
        associatedLabels.push(labels[i]);
    }

    if (element.parentNode instanceof HTMLLabelElement)
      associatedLabels.push(element.parentNode);

    return associatedLabels.filter(this._isElementVisible);
  },

  _isElementVisible: function(element) {
    let style = element.ownerDocument.defaultView.getComputedStyle(element, null);
    let isVisible = (style.getPropertyValue("visibility") != "hidden");
    let isOpaque = (style.getPropertyValue("opacity") != 0);

    let rect = element.getBoundingClientRect();
    return isVisible && isOpaque && (rect.height != 0 || rect.width != 0);
  }
};


/** Implements common choice interface for HTML select elements. */
function SelectWrapper(aControl) {
  this._control = aControl;
}

SelectWrapper.prototype = {
  getSelectedIndex: function() { return this._control.selectedIndex; },
  getMultiple: function() { return this._control.multiple; },
  getOptions: function() { return this._control.options; },
  getChildren: function() { return this._control.children; },

  getText: function(aChild) { return aChild.text; },
  isOption: function(aChild) { return aChild instanceof HTMLOptionElement; },
  isGroup: function(aChild) { return aChild instanceof HTMLOptGroupElement; },
  select: function(aIndex, aSelected, aClearAll) {
    let selectElement = this._control.QueryInterface(Ci.nsISelectElement);
    selectElement.setOptionsSelectedByIndex(aIndex, aIndex, aSelected, aClearAll, false, true);
  },
  focus: function() { this._control.focus(); },
  fireOnChange: function() {
    let control = this._control;
    let evt = this._control.ownerDocument.createEvent("Events");
    evt.initEvent("change", true, true, this._control.ownerDocument.defaultView, 0,
                  false, false,
                  false, false, null);
    new Util.Timeout(function() {
      control.dispatchEvent(evt);
    }).once(0);
  }
};


// Use wrappedJSObject when control is in content for extra protection
// See bug 559792

/** Implements common choice interface for XUL menulist elements. */
function MenulistWrapper(aControl) {
  this._control = aControl;
}

MenulistWrapper.prototype = {
  getSelectedIndex: function() {
    let control = this._control.wrappedJSObject || this._control;
    let result = control.selectedIndex
    return ((typeof result == "number" && !isNaN(result)) ? result : -1);
  },

  getMultiple: function() { return false; },

  getOptions: function() {
    let control = this._control.wrappedJSObject || this._control;
    return control.menupopup.children;
  },

  getChildren: function() {
    let control = this._control.wrappedJSObject || this._control;
    return control.menupopup.children;
  },

  getText: function(aChild) { return aChild.label; },
  isOption: function(aChild) { return aChild instanceof Ci.nsIDOMXULSelectControlItemElement; },
  isGroup: function(aChild) { return false },
  select: function(aIndex, aSelected, aClearAll) {
    let control = this._control.wrappedJSObject || this._control;
    control.selectedIndex = aIndex;
  },
  focus: function() { this._control.focus(); },
  fireOnChange: function() {
    let control = this._control;
    let evt = document.createEvent("XULCommandEvent");
    evt.initCommandEvent("command", true, true, window, 0,
                         false, false,
                         false, false, null);
    new Util.Timeout(function() {
      control.dispatchEvent(evt);
    }).once(0);
  }
};


/**
 * Simple Point class.
 *
 * Any method that takes an x and y may also take a point.
 */
function Point(x, y) {
  this.set(x, y);
}

Point.prototype = {
  clone: function clone() {
    return new Point(this.x, this.y);
  },

  set: function set(x, y) {
    this.x = x;
    this.y = y;
    return this;
  },
  
  equals: function equals(x, y) {
    return this.x == x && this.y == y;
  },

  toString: function toString() {
    return "(" + this.x + "," + this.y + ")";
  },

  map: function map(f) {
    this.x = f.call(this, this.x);
    this.y = f.call(this, this.y);
    return this;
  },

  add: function add(x, y) {
    this.x += x;
    this.y += y;
    return this;
  },

  subtract: function subtract(x, y) {
    this.x -= x;
    this.y -= y;
    return this;
  },

  scale: function scale(s) {
    this.x *= s;
    this.y *= s;
    return this;
  },

  isZero: function() {
    return this.x == 0 && this.y == 0;
  }
};

(function() {
  function takePointOrArgs(f) {
    return function(arg1, arg2) {
      if (arg2 === undefined)
        return f.call(this, arg1.x, arg1.y);
      else
        return f.call(this, arg1, arg2);
    };
  }

  for each (let f in ['add', 'subtract', 'equals', 'set'])
    Point.prototype[f] = takePointOrArgs(Point.prototype[f]);
})();


/**
 * Rect is a simple data structure for representation of a rectangle supporting
 * many basic geometric operations.
 *
 * NOTE: Since its operations are closed, rectangles may be empty and will report
 * non-positive widths and heights in that case.
 */

function Rect(x, y, w, h) {
  this.left = x;
  this.top = y;
  this.right = x+w;
  this.bottom = y+h;
};

Rect.fromRect = function fromRect(r) {
  return new Rect(r.left, r.top, r.right - r.left, r.bottom - r.top);
}

Rect.prototype = {
  get x() { return this.left; },
  get y() { return this.top; },
  get width() { return this.right - this.left; },
  get height() { return this.bottom - this.top; },
  set x(v) {
    let diff = this.left - v;
    this.left = v;
    this.right -= diff;
  },
  set y(v) {
    let diff = this.top - v;
    this.top = v;
    this.bottom -= diff;
  },
  set width(v) { this.right = this.left + v; },
  set height(v) { this.bottom = this.top + v; },

  isEmpty: function isEmpty() {
    return this.left >= this.right || this.top >= this.bottom;
  },

  setRect: function(x, y, w, h) {
    this.left = x;
    this.top = y;
    this.right = x+w;
    this.bottom = y+h;

    return this;
  },

  setBounds: function(l, t, r, b) {
    this.top = t;
    this.left = l;
    this.bottom = b;
    this.right = r;

    return this;
  },

  equals: function equals(other) {
    return other != null &&
            (this.isEmpty() && other.isEmpty() ||
            this.top == other.top &&
            this.left == other.left &&
            this.bottom == other.bottom &&
            this.right == other.right);
  },

  clone: function clone() {
    return new Rect(this.left, this.top, this.right - this.left, this.bottom - this.top);
  },

  center: function center() {
    if (this.isEmpty())
      throw "Empty rectangles do not have centers";
    return new Point(this.left + (this.right - this.left) / 2,
                          this.top + (this.bottom - this.top) / 2);
  },

  copyFrom: function(other) {
    this.top = other.top;
    this.left = other.left;
    this.bottom = other.bottom;
    this.right = other.right;

    return this;
  },

  translate: function(x, y) {
    this.left += x;
    this.right += x;
    this.top += y;
    this.bottom += y;

    return this;
  },

  toString: function() {
    return "[" + this.x + "," + this.y + "," + this.width + "," + this.height + "]";
  },

  /** return a new rect that is the union of that one and this one */
  union: function(other) {
    return this.clone().expandToContain(other);
  },

  contains: function(other) {
    if (other.isEmpty()) return true;
    if (this.isEmpty()) return false;

    return (other.left >= this.left &&
            other.right <= this.right &&
            other.top >= this.top &&
            other.bottom <= this.bottom);
  },

  intersect: function(other) {
    return this.clone().restrictTo(other);
  },

  intersects: function(other) {
    if (this.isEmpty() || other.isEmpty())
      return false;

    let x1 = Math.max(this.left, other.left);
    let x2 = Math.min(this.right, other.right);
    let y1 = Math.max(this.top, other.top);
    let y2 = Math.min(this.bottom, other.bottom);
    return x1 < x2 && y1 < y2;
  },

  /** Restrict area of this rectangle to the intersection of both rectangles. */
  restrictTo: function restrictTo(other) {
    if (this.isEmpty() || other.isEmpty())
      return this.setRect(0, 0, 0, 0);

    let x1 = Math.max(this.left, other.left);
    let x2 = Math.min(this.right, other.right);
    let y1 = Math.max(this.top, other.top);
    let y2 = Math.min(this.bottom, other.bottom);
    // If width or height is 0, the intersection was empty.
    return this.setRect(x1, y1, Math.max(0, x2 - x1), Math.max(0, y2 - y1));
  },

  /** Expand this rectangle to the union of both rectangles. */
  expandToContain: function expandToContain(other) {
    if (this.isEmpty()) return this.copyFrom(other);
    if (other.isEmpty()) return this;

    let l = Math.min(this.left, other.left);
    let r = Math.max(this.right, other.right);
    let t = Math.min(this.top, other.top);
    let b = Math.max(this.bottom, other.bottom);
    return this.setRect(l, t, r-l, b-t);
  },

  /**
   * Expands to the smallest rectangle that contains original rectangle and is bounded
   * by lines with integer coefficients.
   */
  expandToIntegers: function round() {
    this.left = Math.floor(this.left);
    this.top = Math.floor(this.top);
    this.right = Math.ceil(this.right);
    this.bottom = Math.ceil(this.bottom);
    return this;
  },

  scale: function scale(xscl, yscl) {
    this.left *= xscl;
    this.right *= xscl;
    this.top *= yscl;
    this.bottom *= yscl;
    return this;
  },

  map: function map(f) {
    this.left = f.call(this, this.left);
    this.top = f.call(this, this.top);
    this.right = f.call(this, this.right);
    this.bottom = f.call(this, this.bottom);
    return this;
  },

  /** Ensure this rectangle is inside the other, if possible. Preserves w, h. */
  translateInside: function translateInside(other) {
    let offsetX = (this.left < other.left ? other.left - this.left :
        (this.right > other.right ? other.right - this.right : 0));
    let offsetY = (this.top < other.top ? other.top - this.top :
        (this.bottom > other.bottom ? other.bottom - this.bottom : 0));
    return this.translate(offsetX, offsetY);
  },

  /** Subtract other area from this. Returns array of rects whose union is this-other. */
  subtract: function subtract(other) {
    let r = new Rect(0, 0, 0, 0);
    let result = [];
    other = other.intersect(this);
    if (other.isEmpty())
      return [this.clone()];

    // left strip
    r.setBounds(this.left, this.top, other.left, this.bottom);
    if (!r.isEmpty())
      result.push(r.clone());
    // inside strip
    r.setBounds(other.left, this.top, other.right, other.top);
    if (!r.isEmpty())
      result.push(r.clone());
    r.setBounds(other.left, other.bottom, other.right, this.bottom);
    if (!r.isEmpty())
      result.push(r.clone());
    // right strip
    r.setBounds(other.right, this.top, this.right, this.bottom);
    if (!r.isEmpty())
      result.push(r.clone());

    return result;
  },
};
