/*
 * Decompiled with CFR 0.152.
 */
package org.mozilla.jss.pkcs11;

import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.PublicKey;
import java.security.interfaces.DSAPublicKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.AlgorithmParameterSpec;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.RC2ParameterSpec;
import org.mozilla.jss.crypto.Algorithm;
import org.mozilla.jss.crypto.EncryptionAlgorithm;
import org.mozilla.jss.crypto.IVParameterSpec;
import org.mozilla.jss.crypto.KeyPairAlgorithm;
import org.mozilla.jss.crypto.KeyWrapAlgorithm;
import org.mozilla.jss.crypto.KeyWrapper;
import org.mozilla.jss.crypto.PrivateKey;
import org.mozilla.jss.crypto.SymmetricKey;
import org.mozilla.jss.crypto.TokenException;
import org.mozilla.jss.pkcs11.KeyType;
import org.mozilla.jss.pkcs11.PK11PrivKey;
import org.mozilla.jss.pkcs11.PK11PubKey;
import org.mozilla.jss.pkcs11.PK11SymKey;
import org.mozilla.jss.pkcs11.PK11Token;
import org.mozilla.jss.util.Assert;

final class PK11KeyWrapper
implements KeyWrapper {
    private PK11Token token;
    private KeyWrapAlgorithm algorithm;
    private int state = 0;
    private AlgorithmParameterSpec parameters = null;
    private SymmetricKey symKey = null;
    private PrivateKey privKey = null;
    private PublicKey pubKey = null;
    private byte[] IV = null;
    private static final int UNINITIALIZED = 0;
    private static final int WRAP = 1;
    private static final int UNWRAP = 2;

    private PK11KeyWrapper() {
    }

    PK11KeyWrapper(PK11Token token, KeyWrapAlgorithm algorithm) {
        this.token = token;
        this.algorithm = algorithm;
    }

    public void initWrap(SymmetricKey wrappingKey, AlgorithmParameterSpec parameters) throws InvalidKeyException, InvalidAlgorithmParameterException {
        this.initWrap(parameters);
        this.checkWrapper(wrappingKey);
        this.symKey = wrappingKey;
    }

    public void initWrap(PublicKey wrappingKey, AlgorithmParameterSpec parameters) throws InvalidKeyException, InvalidAlgorithmParameterException {
        this.initWrap(parameters);
        this.checkWrapper(wrappingKey);
        this.pubKey = wrappingKey;
    }

    public void initWrap() throws InvalidKeyException, InvalidAlgorithmParameterException {
        if (this.algorithm != KeyWrapAlgorithm.PLAINTEXT) {
            throw new InvalidKeyException(this.algorithm + " requires a key");
        }
        this.reset();
        this.state = 1;
    }

    private void initWrap(AlgorithmParameterSpec parameters) throws InvalidAlgorithmParameterException {
        this.reset();
        this.checkParams(parameters);
        this.parameters = parameters;
        this.state = 1;
    }

    public void initUnwrap(PrivateKey unwrappingKey, AlgorithmParameterSpec parameters) throws InvalidKeyException, InvalidAlgorithmParameterException {
        this.initUnwrap(parameters);
        this.checkWrapper(unwrappingKey);
        this.privKey = unwrappingKey;
    }

    public void initUnwrap(SymmetricKey unwrappingKey, AlgorithmParameterSpec parameters) throws InvalidKeyException, InvalidAlgorithmParameterException {
        this.initUnwrap(parameters);
        this.checkWrapper(unwrappingKey);
        this.symKey = unwrappingKey;
    }

    public void initUnwrap() throws InvalidKeyException, InvalidAlgorithmParameterException {
        if (this.algorithm != KeyWrapAlgorithm.PLAINTEXT) {
            throw new InvalidKeyException(this.algorithm + " requires a key");
        }
        this.reset();
        this.state = 2;
    }

    private void initUnwrap(AlgorithmParameterSpec parameters) throws InvalidAlgorithmParameterException {
        this.reset();
        this.checkParams(parameters);
        this.parameters = parameters;
        this.state = 2;
    }

    private void checkWrapper(PublicKey key) throws InvalidKeyException {
        if (key == null) {
            throw new InvalidKeyException("Key is null");
        }
        if (!(key instanceof PK11PubKey)) {
            throw new InvalidKeyException("Key is not a PKCS #11 key");
        }
        try {
            KeyType type = KeyType.getKeyTypeFromAlgorithm(this.algorithm);
            if (type == KeyType.RSA && !(key instanceof RSAPublicKey) || type == KeyType.DSA && !(key instanceof DSAPublicKey)) {
                throw new InvalidKeyException("Key is not the right type for this algorithm");
            }
        }
        catch (NoSuchAlgorithmException e) {
            Assert.notReached("unable to find algorithm from key type");
        }
    }

    private void checkWrapper(SymmetricKey key) throws InvalidKeyException {
        if (key == null) {
            throw new InvalidKeyException("Key is null");
        }
        if (!((Object)key.getOwningToken()).equals(this.token)) {
            throw new InvalidKeyException("Key does not reside on the current token");
        }
        if (!(key instanceof PK11SymKey)) {
            throw new InvalidKeyException("Key is not a PKCS #11 key");
        }
        try {
            if (((PK11SymKey)key).getKeyType() != KeyType.getKeyTypeFromAlgorithm(this.algorithm)) {
                throw new InvalidKeyException("Key is not the right type for this algorithm");
            }
        }
        catch (NoSuchAlgorithmException e) {
            Assert.notReached("Unknown algorithm");
        }
    }

    private void checkWrapper(PrivateKey key) throws InvalidKeyException {
        if (key == null) {
            throw new InvalidKeyException("Key is null");
        }
        if (!((Object)key.getOwningToken()).equals(this.token)) {
            throw new InvalidKeyException("Key does not reside on the current token");
        }
        if (!(key instanceof PK11PrivKey)) {
            throw new InvalidKeyException("Key is not a PKCS #11 key");
        }
        try {
            if (((PK11PrivKey)key).getKeyType() != KeyType.getKeyTypeFromAlgorithm(this.algorithm)) {
                throw new InvalidKeyException("Key is not the right type for this algorithm");
            }
        }
        catch (NoSuchAlgorithmException e) {
            Assert.notReached("Unknown algorithm");
        }
    }

    private void checkParams(AlgorithmParameterSpec params) throws InvalidAlgorithmParameterException {
        if (!this.algorithm.isValidParameterObject(params)) {
            String name = "null";
            if (params != null) {
                name = params.getClass().getName();
            }
            throw new InvalidAlgorithmParameterException(this.algorithm + " cannot use a " + name + " parameter");
        }
        if (params instanceof IVParameterSpec) {
            this.IV = ((IVParameterSpec)params).getIV();
        } else if (params instanceof IvParameterSpec) {
            this.IV = ((IvParameterSpec)params).getIV();
        } else if (params instanceof RC2ParameterSpec) {
            this.IV = ((RC2ParameterSpec)params).getIV();
        }
    }

    public byte[] wrap(PrivateKey toBeWrapped) throws InvalidKeyException, IllegalStateException, TokenException {
        if (this.state != 1) {
            throw new IllegalStateException();
        }
        if (this.algorithm == KeyWrapAlgorithm.PLAINTEXT) {
            throw new InvalidKeyException("plaintext wrapping not supported");
        }
        this.checkWrappee(toBeWrapped);
        if (this.symKey != null) {
            Assert._assert(this.privKey == null && this.pubKey == null);
            return PK11KeyWrapper.nativeWrapPrivWithSym(this.token, toBeWrapped, this.symKey, this.algorithm, this.IV);
        }
        throw new InvalidKeyException("Wrapping a private key with a public key is not supported");
    }

    public byte[] wrap(SymmetricKey toBeWrapped) throws InvalidKeyException, IllegalStateException, TokenException {
        if (this.state != 1) {
            throw new IllegalStateException();
        }
        if (this.algorithm == KeyWrapAlgorithm.PLAINTEXT) {
            throw new InvalidKeyException("plaintext wrapping not supported");
        }
        this.checkWrappee(toBeWrapped);
        if (this.symKey != null) {
            Assert._assert(this.privKey == null && this.pubKey == null);
            return PK11KeyWrapper.nativeWrapSymWithSym(this.token, toBeWrapped, this.symKey, this.algorithm, this.IV);
        }
        Assert._assert(this.pubKey != null && this.privKey == null && this.symKey == null);
        return PK11KeyWrapper.nativeWrapSymWithPub(this.token, toBeWrapped, this.pubKey, this.algorithm, this.IV);
    }

    private void checkWrappee(SymmetricKey symKey) throws InvalidKeyException {
        if (symKey == null) {
            throw new InvalidKeyException("key to be wrapped is null");
        }
        if (!(symKey instanceof PK11SymKey)) {
            throw new InvalidKeyException("key to be wrapped is not a PKCS #11 key");
        }
        if (!((Object)symKey.getOwningToken()).equals(this.token)) {
            throw new InvalidKeyException("key to be wrapped does not live on the same token as the wrapping key");
        }
    }

    private void checkWrappee(PrivateKey privKey) throws InvalidKeyException {
        if (privKey == null) {
            throw new InvalidKeyException("key to be wrapped is null");
        }
        if (!(privKey instanceof PK11PrivKey)) {
            throw new InvalidKeyException("key to be wrapped is not a PKCS #11 key");
        }
        if (!((Object)privKey.getOwningToken()).equals(this.token)) {
            throw new InvalidKeyException("key to be wrapped does not live on the same token as the wrapping key");
        }
    }

    private static native byte[] nativeWrapSymWithSym(PK11Token var0, SymmetricKey var1, SymmetricKey var2, KeyWrapAlgorithm var3, byte[] var4) throws TokenException;

    private static native byte[] nativeWrapSymWithPub(PK11Token var0, SymmetricKey var1, PublicKey var2, KeyWrapAlgorithm var3, byte[] var4) throws TokenException;

    private static native byte[] nativeWrapPrivWithSym(PK11Token var0, PrivateKey var1, SymmetricKey var2, KeyWrapAlgorithm var3, byte[] var4) throws TokenException;

    public PrivateKey unwrapPrivate(byte[] wrapped, PrivateKey.Type type, PublicKey publicKey) throws TokenException, InvalidKeyException, IllegalStateException {
        return this.baseUnwrapPrivate(wrapped, type, publicKey, false);
    }

    public PrivateKey unwrapTemporaryPrivate(byte[] wrapped, PrivateKey.Type type, PublicKey publicKey) throws TokenException, InvalidKeyException, IllegalStateException {
        return this.baseUnwrapPrivate(wrapped, type, publicKey, true);
    }

    private PrivateKey baseUnwrapPrivate(byte[] wrapped, PrivateKey.Type type, PublicKey publicKey, boolean temporary) throws TokenException, InvalidKeyException, IllegalStateException {
        if (this.state != 2) {
            throw new IllegalStateException();
        }
        if (this.algorithm == KeyWrapAlgorithm.PLAINTEXT) {
            throw new TokenException("plaintext unwrapping of private keys is not supported");
        }
        byte[] publicValue = PK11KeyWrapper.extractPublicValue(publicKey, type);
        if (this.symKey != null) {
            Assert._assert(this.pubKey == null && this.privKey == null);
            return PK11KeyWrapper.nativeUnwrapPrivWithSym(this.token, this.symKey, wrapped, this.algorithm, PK11KeyWrapper.algFromType(type), publicValue, this.IV, temporary);
        }
        throw new InvalidKeyException("Unwrapping a private key with a private key is not supported");
    }

    private static byte[] extractPublicValue(PublicKey publicKey, PrivateKey.Type type) throws InvalidKeyException {
        if (publicKey == null) {
            throw new InvalidKeyException("publicKey is null");
        }
        if (type == PrivateKey.RSA) {
            if (!(publicKey instanceof RSAPublicKey)) {
                throw new InvalidKeyException("Type of public key does not match type of private key");
            }
            return ((RSAPublicKey)publicKey).getModulus().toByteArray();
        }
        if (type == PrivateKey.DSA) {
            if (!(publicKey instanceof DSAPublicKey)) {
                throw new InvalidKeyException("Type of public key does not match type of private key");
            }
            return ((DSAPublicKey)publicKey).getY().toByteArray();
        }
        Assert.notReached("Unknown private key type");
        return new byte[0];
    }

    public SymmetricKey unwrapSymmetric(byte[] wrapped, SymmetricKey.Type type, SymmetricKey.Usage usage, int keyLen) throws TokenException, IllegalStateException, InvalidAlgorithmParameterException {
        return this.unwrapSymmetric(wrapped, type, usage.getVal(), keyLen);
    }

    public SymmetricKey unwrapSymmetric(byte[] wrapped, SymmetricKey.Type type, int keyLen) throws TokenException, IllegalStateException, InvalidAlgorithmParameterException {
        return this.unwrapSymmetric(wrapped, type, -1, keyLen);
    }

    private SymmetricKey unwrapSymmetric(byte[] wrapped, SymmetricKey.Type type, int usageEnum, int keyLen) throws TokenException, IllegalStateException, InvalidAlgorithmParameterException {
        if (this.state != 2) {
            throw new IllegalStateException();
        }
        if (!this.algorithm.isPadded() && type == SymmetricKey.RC4) {
            if (keyLen <= 0) {
                throw new InvalidAlgorithmParameterException("RC4 keys wrapped in unpadded algorithms need key length specified when unwrapping");
            }
        } else {
            keyLen = 0;
        }
        if (this.algorithm == KeyWrapAlgorithm.PLAINTEXT) {
            return PK11KeyWrapper.nativeUnwrapSymPlaintext(this.token, wrapped, PK11KeyWrapper.algFromType(type), usageEnum);
        }
        if (this.symKey != null) {
            Assert._assert(this.pubKey == null && this.privKey == null);
            return PK11KeyWrapper.nativeUnwrapSymWithSym(this.token, this.symKey, wrapped, this.algorithm, PK11KeyWrapper.algFromType(type), keyLen, this.IV, usageEnum);
        }
        Assert._assert(this.privKey != null && this.pubKey == null && this.symKey == null);
        return PK11KeyWrapper.nativeUnwrapSymWithPriv(this.token, this.privKey, wrapped, this.algorithm, PK11KeyWrapper.algFromType(type), keyLen, this.IV, usageEnum);
    }

    private static Algorithm algFromType(PrivateKey.Type type) {
        if (type == PrivateKey.RSA) {
            return KeyPairAlgorithm.RSAFamily;
        }
        if (type == PrivateKey.DSA) {
            return KeyPairAlgorithm.DSAFamily;
        }
        Assert._assert(type == PrivateKey.EC);
        return KeyPairAlgorithm.ECFamily;
    }

    private static Algorithm algFromType(SymmetricKey.Type type) {
        if (type == SymmetricKey.DES) {
            return EncryptionAlgorithm.DES_ECB;
        }
        if (type == SymmetricKey.DES3) {
            return EncryptionAlgorithm.DES3_ECB;
        }
        if (type == SymmetricKey.AES) {
            return EncryptionAlgorithm.AES_128_ECB;
        }
        if (type == SymmetricKey.RC4) {
            return EncryptionAlgorithm.RC4;
        }
        Assert._assert(type == SymmetricKey.RC2);
        return EncryptionAlgorithm.RC2_CBC;
    }

    private static native PrivateKey nativeUnwrapPrivWithSym(PK11Token var0, SymmetricKey var1, byte[] var2, KeyWrapAlgorithm var3, Algorithm var4, byte[] var5, byte[] var6, boolean var7) throws TokenException;

    private static native SymmetricKey nativeUnwrapSymWithSym(PK11Token var0, SymmetricKey var1, byte[] var2, KeyWrapAlgorithm var3, Algorithm var4, int var5, byte[] var6, int var7) throws TokenException;

    private static native SymmetricKey nativeUnwrapSymWithPriv(PK11Token var0, PrivateKey var1, byte[] var2, KeyWrapAlgorithm var3, Algorithm var4, int var5, byte[] var6, int var7) throws TokenException;

    private static native SymmetricKey nativeUnwrapSymPlaintext(PK11Token var0, byte[] var1, Algorithm var2, int var3);

    private void reset() {
        this.state = 0;
        this.symKey = null;
        this.privKey = null;
        this.pubKey = null;
        this.parameters = null;
        this.IV = null;
    }
}

