
// Dependencies: 
// gPrefBranch, gBrandBundle, gMessengerBundle should already be defined
// gatherTextUnder from utilityOverlay.js

const kPhishingNotSuspicious = 0;
const kPhishingWithIPAddress = 1;
const kPhishingWithMismatchedHosts = 2;

//////////////////////////////////////////////////////////////////////////////
// isPhishingURL --> examines the passed in linkNode and returns true if we think
//                   the URL is an email scam.
// aLinkNode: the link node to examine
// aSilentMode: don't prompt the user to confirm
//////////////////////////////////////////////////////////////////////////////

function isPhishingURL(aLinkNode, aSilentMode)
{
  if (!gPrefBranch.getBoolPref("mail.phishing.detection.enabled"))
    return false;

  var phishingType = kPhishingNotSuspicious;
  var href = aLinkNode.href;
  var linkTextURL = {};

  var ioService = Components.classes["@mozilla.org/network/io-service;1"].getService(Components.interfaces.nsIIOService);
  hrefURL  = ioService.newURI(href, null, null);
  
  // (1) if the host name is an IP address then block the url...
  // TODO: add support for IPv6
  var ipv4HostRegExp = new RegExp(/\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}/);  // IPv4
  if (ipv4HostRegExp.test(hrefURL.host))
    phishingType = kPhishingWithIPAddress;
  else if (misMatchedHostWithLinkText(aLinkNode, hrefURL, linkTextURL))
    phishingType = kPhishingWithMismatchedHosts;

  var isPhishingURL = phishingType != kPhishingNotSuspicious;

  if (!aSilentMode && isPhishingURL) // allow the user to over ride the decision
    isPhishingURL = confirmSuspiciousURL(phishingType, hrefURL);

  return isPhishingURL;
}

//////////////////////////////////////////////////////////////////////////////
// helper methods in support of isPhishingURL
//////////////////////////////////////////////////////////////////////////////

function misMatchedHostWithLinkText(aLinkNode, aHrefURL, aLinkTextURL)
{
  var linkNodeText = gatherTextUnder(aLinkNode);    
  // only worry about http and https urls
  if (linkNodeText && (aHrefURL.schemeIs('http') || aHrefURL.schemeIs('https')))
  {
    // does the link text look like a http url?
     if (linkNodeText.search(/(^http:|^https:)/) != -1)
     {
       var ioService = Components.classes["@mozilla.org/network/io-service;1"].getService(Components.interfaces.nsIIOService);
       linkTextURL  = ioService.newURI(linkNodeText, null, null);
       aLinkTextURL.value = linkTextURL;
       return aHrefURL.host != linkTextURL.host;
     }
  }

  return false;
}

// returns true if the user confirms the URL is a scam
function confirmSuspiciousURL(phishingType, hrefURL)
{
  var brandShortName = gBrandBundle.getString("brandRealShortName");
  var titleMsg = gMessengerBundle.getString("confirmPhishingTitle");
  var dialogMsg;

  switch (phishingType)
  {
    case kPhishingWithIPAddress:
    case kPhishingWithMismatchedHosts:
      dialogMsg = gMessengerBundle.getFormattedString("confirmPhishingUrl" + phishingType, [brandShortName, hrefURL.host], 2);
      break;
    default:
      return false;
  }

  const nsIPS = Components.interfaces.nsIPromptService;
  var promptService = Components.classes["@mozilla.org/embedcomp/prompt-service;1"].getService(nsIPS);
  var buttons = nsIPS.STD_YES_NO_BUTTONS + nsIPS.BUTTON_POS_1_DEFAULT;
  return promptService.confirmEx(window, titleMsg, dialogMsg, buttons, "", "", "", "", {}); /* the yes button is in position 0 */
}
